<?php
require_once '../includes/config.php';
require_once __DIR__ . '/includes/admin_auth.php';
require_once __DIR__ . '/includes/admin_header.php';

if (!isset($_GET['id'])) {
    header("Location: orders.php");
    exit();
}

$order_id = intval($_GET['id']);

// Fetch order and customer details
$stmt = $conn->prepare("
    SELECT o.*, c.name as customer_name, c.email, c.phone, c.address 
    FROM orders o
    JOIN customers c ON o.customer_id = c.customer_id
    WHERE o.order_id = ?
");
$stmt->bind_param("i", $order_id);
$stmt->execute();
$result = $stmt->get_result();
$order = $result->fetch_assoc();

if (!$order) {
    header("Location: orders.php");
    exit();
}
$stmt->close();

// Fetch order items
$stmt = $conn->prepare("
    SELECT oi.*, p.name as product_name, p.image_url, 
           IFNULL(v.variant_name, 'N/A') as variant_name
    FROM order_items oi
    LEFT JOIN products p ON oi.product_id = p.product_id
    LEFT JOIN product_variants v ON oi.variant_id = v.variant_id
    WHERE oi.order_id = ?
");
$stmt->bind_param("i", $order_id);
$stmt->execute();
$items_result = $stmt->get_result();

$items = [];
while ($row = $items_result->fetch_assoc()) {
    $items[] = $row;
}
$stmt->close();

require_once __DIR__ . '/includes/admin_header.php';
?>

<div class="admin-content">
    <div class="page-header">
        <h1 class="page-title">Order #<?= $order['order_id'] ?></h1>
        <div class="breadcrumb">
            <li><a href="orders.php">Orders</a></li>
            <li>View</li>
        </div>
    </div>

    <div class="form-grid">
        <div class="form-container">
            <h3>Order Details</h3>
            <div class="form-group">
                <label>Order Status</label>
                <div class="status-badge <?= $order['status'] ?> large">
                    <?= ucfirst($order['status']) ?>
                </div>
            </div>
            <div class="form-group">
                <label>Order Date</label>
                <p><?= date('F j, Y, g:i a', strtotime($order['order_date'])) ?></p>
            </div>
            <div class="form-group">
                <label>Customer Notes</label>
                <p><?= $order['notes'] ? nl2br(htmlspecialchars($order['notes'])) : 'No notes' ?></p>
            </div>

            <form action="order_update_status.php" method="post">
                <input type="hidden" name="id" value="<?= $order['order_id'] ?>">
                <div class="form-group">
                    <label for="status">Update Status</label>
                    <select id="status" name="status" class="form-control">
                        <option value="pending" <?= $order['status'] == 'pending' ? 'selected' : '' ?>>Pending</option>
                        <option value="processing" <?= $order['status'] == 'processing' ? 'selected' : '' ?>>Processing</option>
                        <option value="completed" <?= $order['status'] == 'completed' ? 'selected' : '' ?>>Completed</option>
                        <option value="cancelled" <?= $order['status'] == 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                    </select>
                </div>
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Update Status</button>
                </div>
            </form>
        </div>

        <div class="form-container">
            <h3>Customer Information</h3>
            <div class="form-group">
                <label>Customer Name</label>
                <p><?= htmlspecialchars($order['customer_name']) ?></p>
            </div>
            <div class="form-group">
                <label>Email</label>
                <p><?= htmlspecialchars($order['email']) ?></p>
            </div>
            <div class="form-group">
                <label>Phone</label>
                <p><?= htmlspecialchars($order['phone']) ?></p>
            </div>
            <div class="form-group">
                <label>Delivery Address</label>
                <p><?= nl2br(htmlspecialchars($order['address'])) ?></p>
            </div>
        </div>
    </div>

    <div class="data-table-container" style="margin-top: 2rem;">
        <h3>Order Items</h3>
        <table class="data-table">
            <thead>
                <tr>
                    <th>Product</th>
                    <th>Variant</th>
                    <th>Price</th>
                    <th>Quantity</th>
                    <th>Subtotal</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($items as $item): ?>
                <tr>
                    <td>
                        <div class="user-cell">
                            <?php if ($item['image_url']): ?>
                            <img src="../images/products/<?= htmlspecialchars($item['image_url']) ?>" alt="<?= htmlspecialchars($item['product_name']) ?>">
                            <?php endif; ?>
                            <span><?= htmlspecialchars($item['product_name']) ?></span>
                        </div>
                    </td>
                    <td><?= htmlspecialchars($item['variant_name']) ?></td>
                    <td>KSh <?= number_format($item['price'], 2) ?></td>
                    <td><?= $item['quantity'] ?></td>
                    <td>KSh <?= number_format($item['price'] * $item['quantity'], 2) ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr>
                    <td colspan="4" style="text-align: right;"><strong>Subtotal</strong></td>
                    <td>KSh <?= number_format($order['total_amount'], 2) ?></td>
                </tr>
                <tr>
                    <td colspan="4" style="text-align: right;"><strong>Delivery</strong></td>
                    <td>FREE</td>
                </tr>
                <tr>
                    <td colspan="4" style="text-align: right;"><strong>Total</strong></td>
                    <td>KSh <?= number_format($order['total_amount'], 2) ?></td>
                </tr>
            </tfoot>
        </table>
    </div>

    <div class="form-actions" style="margin-top: 2rem;">
        <a href="order_invoice.php?id=<?= $order['order_id'] ?>" class="btn" target="_blank">
            <i class="fas fa-file-invoice"></i> View Invoice
        </a>
        <a href="orders.php" class="btn">
            <i class="fas fa-arrow-left"></i> Back to Orders
        </a>
    </div>
</div>

<?php require_once __DIR__ . '/includes/admin_footer.php'; ?>
